const { TranslationServiceClient } = require('@google-cloud/translate').v3;
const config = require('../config/config');
const countryLanguages = require('./countryLanguages');
const countryCodes = require('./CountryCodes.json');
const locales = require('../config/locales.json');

class Translator {
  constructor() {
    this.client = null;
    this.projectId = config.translation.projectId;
    this.location = 'global';
    this.initialized = false;
    this.countryCodes = countryCodes;
    this.locales = locales;
    
    // Create dynamic mapping: Country Code → Primary Language
    this.countryToLanguageMap = this.buildCountryLanguageMap();
  }
  
  /**
   * Build country to language mapping using locales.json
   */
  buildCountryLanguageMap() {
    const map = {};
    
    // Iterate over all locales
    this.locales.forEach(locale => {
      const countryCode = locale.country?.code;
      const languageCode = locale.language?.iso_639_1;
      
      if (countryCode && languageCode) {
        // If country doesn't have language mapped yet, add the first one
        if (!map[countryCode]) {
          map[countryCode] = languageCode;
        }
        // Prioritize more common languages
        const priorityLanguages = ['en', 'pt', 'es', 'fr', 'de', 'it', 'zh', 'ja', 'ko', 'ru', 'ar', 'hi'];
        if (priorityLanguages.includes(languageCode)) {
          map[countryCode] = languageCode;
        }
      }
    });
    
    console.log(`✅ Language mapping loaded: ${Object.keys(map).length} countries`);
    return map;
  }

  /**
   * Initialize Google translation client
   */
  async initialize() {
    try {
      if (config.translation.projectId && config.translation.credentials) {
        this.client = new TranslationServiceClient({
          keyFilename: config.translation.credentials
        });
        this.initialized = true;
        console.log('✅ Google Translate initialized successfully');
      } else {
        console.warn('⚠️ Google Translate not configured - using simulated mode');
        this.initialized = false;
      }
    } catch (error) {
      console.error('❌ Error initializing Google Translate:', error.message);
      this.initialized = false;
    }
  }

  /**
   * Translate text from one language to another
   * @param {string} text - Text to be translated
   * @param {string} targetLanguage - Target language (ISO code)
   * @param {string} sourceLanguage - Source language (optional)
   * @returns {Promise<object>} - {translatedText, detectedLanguage}
   */
  async translate(text, targetLanguage, sourceLanguage = null) {
    if (!text || text.trim() === '') {
      return { translatedText: text, detectedLanguage: targetLanguage };
    }

    // Validate target language
    if (!targetLanguage) {
      console.warn(`⚠️ Language not specified, using 'en'`);
      targetLanguage = 'en';
    }

    // If not initialized, return original text
    if (!this.initialized || !this.client) {
      return {
        translatedText: text,
        detectedLanguage: sourceLanguage || 'unknown'
      };
    }

    try {
      const request = {
        parent: `projects/${this.projectId}/locations/${this.location}`,
        contents: [text],
        targetLanguageCode: targetLanguage,
        mimeType: 'text/plain'
      };

      if (sourceLanguage) {
        request.sourceLanguageCode = sourceLanguage;
      }

      const [response] = await this.client.translateText(request);
      const translation = response.translations[0];

      return {
        translatedText: translation.translatedText,
        detectedLanguage: translation.detectedLanguageCode || sourceLanguage
      };
    } catch (error) {
      console.error('❌ Erro ao traduzir:', error.message);
      return {
        translatedText: text,
        detectedLanguage: sourceLanguage || 'unknown'
      };
    }
  }

  /**
   * Detecta o idioma de um texto
   * @param {string} text - Texto para detectar idioma
   * @returns {Promise<string>} - Código do idioma detectado
   */
  async detectLanguage(text) {
    if (!text || text.trim() === '') {
      return 'unknown';
    }

    if (!this.initialized || !this.client) {
      return 'unknown';
    }

    try {
      const request = {
        parent: `projects/${this.projectId}/locations/${this.location}`,
        content: text,
        mimeType: 'text/plain'
      };

      const [response] = await this.client.detectLanguage(request);
      const detection = response.languages[0];
      
      return detection.languageCode;
    } catch (error) {
      console.error('❌ Erro ao detectar idioma:', error.message);
      return 'unknown';
    }
  }

  /**
   * Protect commands from translation
   * @param {string} text - Text that may contain commands
   * @returns {string} - Text with protected commands
   */
  protectCommands(text) {
    // Regex to match commands like /command or /command word
    const commandPattern = /\/[a-z]+(\s+[a-z]+)?/gi;
    const commands = [];
    
    // Extract and replace commands with placeholders
    let protectedText = text.replace(commandPattern, (match) => {
      const index = commands.length;
      commands.push(match);
      return `__CMD${index}__`;
    });
    
    return { protectedText, commands };
  }

  /**
   * Restore commands after translation
   * @param {string} text - Translated text with placeholders
   * @param {Array} commands - Original commands
   * @returns {string} - Text with restored commands
   */
  restoreCommands(text, commands) {
    let restoredText = text;
    commands.forEach((cmd, index) => {
      restoredText = restoredText.replace(`__CMD${index}__`, cmd);
    });
    return restoredText;
  }

  /**
   * Translate message to client (from default language to client language)
   * @param {string} text - Text to translate
   * @param {string} clientLanguage - Client language
   * @returns {Promise<string>} - Translated text
   */
  async translateToClient(text, clientLanguage) {
    console.log(`🌐 translateToClient: "${text.substring(0, 50)}..." from ${config.translation.defaultLanguage} to ${clientLanguage}`);
    
    if (clientLanguage === config.translation.defaultLanguage) {
      console.log(`   ⏭️ Same language, skipping translation`);
      return text;
    }

    try {
      // Protect commands before translation
      const { protectedText, commands } = this.protectCommands(text);
      
      const result = await this.translate(protectedText, clientLanguage, config.translation.defaultLanguage);
      
      // Restore commands after translation
      const finalText = this.restoreCommands(result.translatedText, commands);
      
      console.log(`   ✅ Translated: "${finalText.substring(0, 50)}..."`);
      return finalText;
    } catch (error) {
      console.error(`   ❌ Translation error:`, error.message);
      throw error;
    }
  }

  /**
   * Traduz mensagem do cliente para o idioma da equipe
   * @param {string} text - Texto a ser traduzido
   * @param {string} clientLanguage - Idioma do cliente
   * @returns {Promise<string>} - Texto traduzido
   */
  async translateToTeam(text, clientLanguage) {
    if (clientLanguage === config.translation.defaultLanguage) {
      return text;
    }

    try {
      // Protect commands before translation
      const { protectedText, commands } = this.protectCommands(text);
      
      const result = await this.translate(protectedText, config.translation.defaultLanguage, clientLanguage);
      
      // Restore commands after translation
      return this.restoreCommands(result.translatedText, commands);
    } catch (error) {
      console.error('⚠️ Error translating to team:', error.message);
      return text;
    }
  }

  /**
   * Translate system message to the configured team language
   * @param {string} textEN - Text in English
   * @returns {Promise<string>} - Text translated to team language
   */
  async translateSystemMessage(textEN) {
    const teamLanguage = config.translation.defaultLanguage;
    
    // If team language is EN, return directly
    if (teamLanguage === 'en') {
      return textEN;
    }

    try {
      const result = await this.translate(textEN, teamLanguage, 'en');
      return result.translatedText;
    } catch (error) {
      console.error('⚠️ Error translating system message:', error.message);
      return textEN; // Return in EN as fallback
    }
  }

  /**
   * Identifica idioma pelo código do país (número de telefone)
   * @param {string} phoneNumber - Número de telefone com código do país
   * @returns {string} - Código do idioma identificado
   */
  identifyLanguageByPhone(phoneNumber) {
    // Remove caracteres não numéricos
    const cleanNumber = phoneNumber.replace(/\D/g, '');
    
    // Procura no CountryCodes.json pelo código de discagem que corresponde
    // Ordena por tamanho do dial_code (maior primeiro) para evitar matches parciais incorretos
    const sortedCountries = [...this.countryCodes].sort((a, b) => {
      const aLen = a.dial_code.replace(/\D/g, '').length;
      const bLen = b.dial_code.replace(/\D/g, '').length;
      return bLen - aLen;
    });
    
    for (const country of sortedCountries) {
      const dialCode = country.dial_code.replace(/\D/g, '');
      if (cleanNumber.startsWith(dialCode)) {
        // Encontrou o país, agora busca idioma no mapeamento dinâmico
        const language = this.countryToLanguageMap[country.code];
        if (language) {
          console.log(`🌍 Idioma detectado: ${language} (${country.name}, ${country.dial_code})`);
          return { language, countryName: country.name, dialCode: country.dial_code };
        }
        // País encontrado mas sem mapeamento - usa inglês
        console.log(`⚠️ País ${country.name} sem idioma mapeado, usando inglês`);
        return { language: 'en', countryName: country.name, dialCode: country.dial_code };
      }
    }

    // Retorna idioma padrão se não identificar
    console.log(`⚠️ País não detectado para ${phoneNumber}, usando inglês`);
    return { language: 'en', countryName: 'Unknown', dialCode: '+' };
  }

  /**
   * Gera mensagem de seleção de idioma multilíngue
   * Uses countryLanguages module for comprehensive language list
   * @returns {string} - Mensagem formatada
   */
  getLanguageSelectionMessage() {
    return countryLanguages.getPopularLanguagesFormatted();
  }

  /**
   * Map selection to language code
   * Uses countryLanguages module for parsing
   * @param {string} selection - User selection (number or code)
   * @returns {string|null} - Language code or null
   */
  mapSelectionToLanguage(selection) {
    return countryLanguages.parseLanguageSelection(selection);
  }

  /**
   * Get language name by code
   * @param {string} code - Language code
   * @returns {string} - Language name
   */
  getLanguageName(code) {
    try {
      return countryLanguages.getLanguageName(code);
    } catch (error) {
      return code.toUpperCase();
    }
  }

  /**
   * Traduz mensagem automaticamente para o idioma do cliente
   * @param {string} message - Mensagem em português
   * @param {string} targetLanguage - Idioma de destino
   * @returns {Promise<string>} - Mensagem traduzida
   */
  async translateMessage(message, targetLanguage) {
    // Se já está em português, não traduz
    if (targetLanguage === 'en') {
      return message;
    }

    try {
      // Protect commands before translation
      const { protectedText, commands } = this.protectCommands(message);
      
      const result = await this.translate(protectedText, targetLanguage, 'en');
      
      // Restore commands after translation
      return this.restoreCommands(result.translatedText, commands);
    } catch (error) {
      console.error(`❌ Erro ao traduzir para ${targetLanguage}:`, error.message);
      return message; // Retorna original em caso de erro
    }
  }

  /**
   * Obtém nome do idioma
   * @param {string} languageCode - Código do idioma
   * @returns {string} - Nome do idioma
   */
  getLanguageName(languageCode) {
    return config.languageNames[languageCode] || languageCode.toUpperCase();
  }
}

module.exports = new Translator();
